#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdbool.h>

#include "elemzo.h"

/*
    A NYELVTAN:

    osszeg = szorzat {('+' | '-') szorzat}
    szorzat = tenyezo {('*' | '/') tenyezo}
    tenyezo = cellacim | szam | zarojeles
    zarojeles = '(' osszeg ')'
*/

/* Lexer függvények */
static bool szokoz(char **szoveg);
static bool karakter(char **szoveg, char const *ertekkeszlet, char *talalat);
static bool szam(char **szoveg, double *ertek);
static bool cellacim(char **szoveg, char *oszlop, int *sor);

/* Parser függvények */
static bool osszeg(char **szoveg, double *ertek, tablazat *a_tablazat);
static bool szorzat(char **szoveg, double *ertek, tablazat *a_tablazat);
static bool tenyezo(char **szoveg, double *ertek, tablazat *a_tablazat);
static bool zarojeles(char **szoveg, double *ertek, tablazat *a_tablazat);


/* Lexer függvények */
static bool szokoz(char **szoveg) {
    while (isspace(**szoveg)) *szoveg += 1;
    return true;
}


static bool karakter(char **szoveg, char const *ertekkeszlet, char *talalat) {
    char *munka = *szoveg;

    szokoz(&munka);

    while (*ertekkeszlet && munka[0] != *ertekkeszlet) {
        ++ertekkeszlet;
    }

    if (*ertekkeszlet == 0) {
        return false;
    }
    else {
        *talalat = *ertekkeszlet;

        *szoveg = munka + 1;
        return true;
    }
}


static bool szam(char **szoveg, double *ertek) {
    char *munka = *szoveg;
    int bajtok;
    double szam;

    szokoz(&munka);
    if (sscanf(munka, "%lf%n", &szam, &bajtok) >= 1) {
        munka += bajtok;
        *ertek = szam;
        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}


static bool cellacim(char **szoveg, char *oszlop, int *sor) {
    char *munka = *szoveg;

    szokoz(&munka);
    if (isalpha(munka[0])) {
        int bajtok;

        *oszlop = toupper(munka[0]);

        munka += 1;
        if (sscanf(munka, "%d%n", sor, &bajtok)>=1) {
            munka += bajtok;
            *szoveg = munka;
            return true;
        }
    }

    return false;
}


/* Parser függvények */
static bool osszeg(char **szoveg, double *ertek, tablazat *a_tablazat) {
    char *munka = *szoveg;
    char kar;
    double ertek_1, ertek_2;

    szokoz(&munka);
    if (szorzat(&munka, &ertek_1, a_tablazat)) {
        while (karakter(&munka, "+-", &kar)) {
            if (szorzat(&munka, &ertek_2, a_tablazat)) {
                ertek_1 = (kar == '+') ? ertek_1 + ertek_2 : ertek_1 - ertek_2;
            }
            else return false;
        }
        *ertek = ertek_1;
        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}


static bool szorzat(char **szoveg, double *ertek, tablazat *a_tablazat) {
    char *munka = *szoveg;
    char kar;
    double ertek_1, ertek_2;

    szokoz(&munka);
    if (tenyezo(&munka, &ertek_1, a_tablazat)) {
        while (karakter(&munka, "*/", &kar)) {
            if (tenyezo(&munka, &ertek_2, a_tablazat)) {
                ertek_1 = (kar == '*') ? ertek_1 * ertek_2 : ertek_1 / ertek_2;
            }
            else return false;
        }
        *ertek = ertek_1;
        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}


static bool tenyezo(char **szoveg, double *ertek, tablazat *a_tablazat) {
    char *munka = *szoveg;
    char oszlop;
    int sor;

    szokoz(&munka);
    if (zarojeles(&munka, ertek, a_tablazat)) {
        *szoveg = munka;
        return true;
    }
    else if (szam(&munka, ertek)) {
        *szoveg = munka;
        return true;
    }
    else if (cellacim(&munka, &oszlop, &sor)) {
        if (cella_kiszamol(a_tablazat, sor, oszlop - 'A')) {
            *ertek = a_tablazat->adat[sor][oszlop - 'A'].cachelt_ertek;
            *szoveg = munka;
            return true;
        }
    }

    return false;
}


static bool zarojeles(char **szoveg, double *ertek, tablazat *a_tablazat) {
    char *munka = *szoveg;
    char kar;

    szokoz(&munka);
    if (karakter(&munka, "(", &kar) && osszeg(&munka, ertek, a_tablazat) && karakter(&munka, ")", &kar)) {
        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}


bool kiertekel(char *szoveg, double *ertek, tablazat *a_tablazat) {
    return osszeg(&szoveg, ertek, a_tablazat);
}
