#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include "ertelmezo.h"


/* Az AST függvényei */
static szimbolum *uj_muvelet(char muveleti_jel);
static szimbolum *uj_szam(double ertek);
static szimbolum *uj_valtozo(char const *nev);
static szimbolum *uj_ertekadas(char const *nev, szimbolum *ast);

/* Lexer függvények */
static bool szokoz(char **szoveg);
static bool karakter(char **szoveg, char const *ertekkeszlet, char *talalat);
static bool szam(char **szoveg, double *ertek);
static bool valtozonev(char **szoveg, char *nev);

/* Parser függvények */
static bool ertekado_kifejezes(char **szoveg, szimbolum **ast);
static bool osszeg(char **szoveg, szimbolum **ast);
static bool szorzat(char **szoveg, szimbolum **ast);
static bool tenyezo(char **szoveg, szimbolum **ast);
static bool zarojeles(char **szoveg, szimbolum **ast);

/* Az elemző egyetlen publikus függvénye */
bool kiertekel(char *szoveg, szimbolum **ast) {
    return ertekado_kifejezes(&szoveg, ast);
}

/* Lexer függvények */
static bool szokoz(char **szoveg) {
    while (isspace(**szoveg)) *szoveg += 1;
    return false;
}

static bool karakter(char **szoveg, char const *ertekkeszlet, char *talalat) {
    char *eleje = *szoveg;
    szokoz(szoveg);

    while (*ertekkeszlet && **szoveg != *ertekkeszlet) {
        ++ertekkeszlet;
    }

    if (*ertekkeszlet == 0) {
        *szoveg = eleje;
        return false;
    }
    else {
        *talalat = *ertekkeszlet;
        *szoveg += 1;
        return true;
    }
}

static bool szam(char **szoveg, double *ertek) {
    char *eleje = *szoveg;
    char kar;

    szokoz(szoveg);
    if (karakter(szoveg, "+-0123456789", &kar)) {
        double szam;
        int bajtok;

        *szoveg -= 1;
        if (sscanf(*szoveg, "%lf%n", &szam, &bajtok) > 0) {
            *szoveg += bajtok;
            *ertek = szam;
            return true;
        }
        else {
            *szoveg = eleje;
            return false;
        }
    }
    else {
        *szoveg = eleje;
        return false;
    }
}

static bool valtozonev(char **szoveg, char *nev) {
    char *munka = *szoveg;
    int hossz = 0;

    szokoz(&munka);
    if (isalpha(munka[hossz++])) {
        while (isalnum(munka[hossz]) && hossz < 50) hossz++;

        if (hossz == 50 && isalnum(munka[hossz])) return false; /* túl hosszú név nem lehet változó */
        else {
            int i;
            for (i = 0; i < hossz; ++i) nev[i] = munka[i];
            nev[i] = 0;

            *szoveg = munka + hossz;
            return true;
        }
    }
    else return false;
}

/* Parser függvények */
static bool osszeg(char **szoveg, szimbolum **ast) {
    char *munka = *szoveg;
    char kar;
    szimbolum *op1, *op2, *uj = NULL;

    szokoz(&munka);
    if (szorzat(&munka, &op1)) {
        while (karakter(&munka, "+-", &kar)) {
            if (szorzat(&munka, &op2)) {
                uj = uj_muvelet(kar);
                uj->op1 = op1;
                uj->op2 = op2;

                op1 = uj;
            }
            else return false;
        }

        if (uj == NULL) *ast = op1;
        else *ast = uj;

        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}

static bool szorzat(char **szoveg, szimbolum **ast) {
    char *munka = *szoveg;
    char kar;
    szimbolum *op1, *op2, *uj = NULL;

    szokoz(&munka);
    if (tenyezo(&munka, &op1)) {
        while (karakter(&munka, "*/", &kar)) {
            if (tenyezo(&munka, &op2)) {
                uj = uj_muvelet(kar);
                uj->op1 = op1;
                uj->op2 = op2;

                op1 = uj;
            }
            else return false;
        }

        if (uj == NULL) *ast = op1;
        else *ast = uj;

        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}

static bool tenyezo(char **szoveg, szimbolum **ast) {
    char *munka = *szoveg;
    double ertek;
    char nev[51];

    szokoz(&munka);
    if (zarojeles(&munka, ast)) {
        *szoveg = munka;
        return true;
    }
    else if (szam(&munka, &ertek)) {
        *ast = uj_szam(ertek);
        *szoveg = munka;
        return true;
    }
    else if (valtozonev(&munka, nev)) {
        *ast = uj_valtozo(nev);
        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }

}

static bool zarojeles(char **szoveg, szimbolum **ast) {
    char *munka = *szoveg;
    char kar;

    szokoz(&munka);
    if (karakter(&munka, "(", &kar) && ertekado_kifejezes(&munka, ast) && karakter(&munka, ")", &kar)) {
        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}

static bool ertekado_kifejezes(char **szoveg, szimbolum **ast) {
    char *munka = *szoveg;
    char nev[51], kar;
    bool van_ertekadas = false;
    szimbolum *kifejezes;

    if (valtozonev(&munka, nev) && karakter(&munka, "=", &kar)) {
        van_ertekadas = true;
    }
    /* Ha nem volt értékadás, vagy nem volt teljes (!), akkor vissza kell állítani a munka pointert: */
    else munka = *szoveg;


    if (osszeg(&munka, &kifejezes)) {
        if (van_ertekadas) {
            *ast = uj_ertekadas(nev, kifejezes);
        }
        else {
            *ast = kifejezes;
        }

        *szoveg = munka;
        return true;
    }
    else {
        return false;
    }
}

/* Az AST függvényei */
static szimbolum *uj_muvelet(char muveleti_jel) {
    szimbolum *uj = (szimbolum *)malloc(sizeof(szimbolum));

    if (uj == NULL) return NULL;

    uj->tipus = MUVELET;
    uj->adat.muveleti_jel = muveleti_jel;
    uj->op1 = uj->op2 = NULL;

    return uj;
}

static szimbolum *uj_valtozo(char const *nev) {
    szimbolum *uj = (szimbolum *)malloc(sizeof(szimbolum));

    if (uj == NULL) return NULL;

    uj->tipus = VALTOZO;
    strncpy(uj->adat.nev, nev, 50); uj->adat.nev[50] = 0;
    uj->op1 = uj->op2 = NULL;

    return uj;
}

static szimbolum *uj_szam(double ertek) {
    szimbolum *uj = (szimbolum *)malloc(sizeof(szimbolum));

    if (uj == NULL) return NULL;

    uj->tipus = SZAM;
    uj->adat.szam = ertek;
    uj->op1 = uj->op2 = NULL;

    return uj;
}

static szimbolum *uj_ertekadas(char const *nev, szimbolum *ast) {
    szimbolum *uj = (szimbolum *)malloc(sizeof(szimbolum));

    if (uj == NULL) return NULL;

    uj->tipus = ERTEKADAS;
    strncpy(uj->adat.egy_ertekadas.nev, nev, 50); uj->adat.egy_ertekadas.nev[50] = 0;
    uj->adat.egy_ertekadas.kifejezes = ast;
    uj->op1 = uj->op2 = NULL;

    return uj;
}

void ast_kiir(szimbolum *ast) {
    if (ast != NULL) {
        ast_kiir(ast->op1);

        if (ast->tipus == SZAM) {
            printf(" %g ", ast->adat.szam);
        }
        else {
            printf(" %c ", ast->adat.muveleti_jel);
        }

        ast_kiir(ast->op2);
    }
}

double ast_kiertekel(szimbolum *ast, valtozo **valtozok) {
    if (ast == NULL) abort();

    switch (ast->tipus) {
        case SZAM:
            return ast->adat.szam;

        case VALTOZO:
            return *lekerdez(valtozok, ast->adat.nev);

        case MUVELET:
        {
            double op1 = ast_kiertekel(ast->op1, valtozok), op2 = ast_kiertekel(ast->op2, valtozok);
            switch (ast->adat.muveleti_jel) {
                case '+':
                    return op1 + op2;
                case '-':
                    return op1 - op2;
                case '*':
                    return op1 * op2;
                case '/':
                    return op1 / op2;
            }
        }

        case ERTEKADAS:
        {
            double op1 = ast_kiertekel(ast->adat.egy_ertekadas.kifejezes, valtozok);
            *lekerdez(valtozok, ast->adat.egy_ertekadas.nev) = op1;
            return op1;
        }
    }

    return 0.0; //default ágak hiánya miatt
}

void ast_torol(szimbolum *ast) {
    if (ast != NULL) {
        ast_torol(ast->op1);
        ast_torol(ast->op2);
        free(ast);
    }
}

